<?php
/**
 * Test Mail System Only - No Database Required
 * Tests the mail system functionality without database connections
 */

echo "=== ZForex Email Integration Test (Mail System Only) ===\n\n";

// Define required constants for mail system testing
if (!defined('MAIL_HOST')) {
    define('MAIL_HOST', 'smtp.gmail.com');
}
if (!defined('MAIL_USERNAME')) {
    define('MAIL_USERNAME', 'test@example.com');
}
if (!defined('MAIL_PASSWORD')) {
    define('MAIL_PASSWORD', 'test_password');
}
if (!defined('MAIL_PORT')) {
    define('MAIL_PORT', 587);
}
if (!defined('MAIL_ENCRYPTION')) {
    define('MAIL_ENCRYPTION', 'tls');
}
if (!defined('MAIL_FROM_EMAIL')) {
    define('MAIL_FROM_EMAIL', 'noreply@example.com');
}
if (!defined('MAIL_FROM_NAME')) {
    define('MAIL_FROM_NAME', 'ZForex Test');
}
if (!defined('EMAIL_AUTOMATION_ENABLED')) {
    define('EMAIL_AUTOMATION_ENABLED', true);
}
if (!defined('LOG_PATH')) {
    define('LOG_PATH', __DIR__ . '/logs/');
}
if (!defined('COMPANY_NAME')) {
    define('COMPANY_NAME', 'ZForex Trading');
}
if (!defined('COMPANY_WEBSITE')) {
    define('COMPANY_WEBSITE', 'https://zforex.com');
}
if (!defined('SUPPORT_EMAIL')) {
    define('SUPPORT_EMAIL', 'support@zforex.com');
}
if (!defined('COMPANY_ADDRESS')) {
    define('COMPANY_ADDRESS', '123 Trading Street, Financial District');
}

// Test 1: Load mail system
echo "1. Testing mail system loading...\n";
try {
    require_once __DIR__ . '/mail_system/integrated_mail_system.php';
    echo "   ✓ Mail system loaded successfully\n";
    $mailSystemLoaded = true;
} catch (Exception $e) {
    echo "   ✗ Failed to load mail system: " . $e->getMessage() . "\n";
    $mailSystemLoaded = false;
}

if (!$mailSystemLoaded) {
    echo "\nTest failed: Cannot proceed without mail system\n";
    exit(1);
}

// Test 2: Check if IntegratedMailSystem class exists
echo "\n2. Testing IntegratedMailSystem class...\n";
if (class_exists('IntegratedMailSystem')) {
    echo "   ✓ IntegratedMailSystem class found\n";
    $classExists = true;
} else {
    echo "   ✗ IntegratedMailSystem class not found\n";
    $classExists = false;
}

// Test 3: Check helper functions
echo "\n3. Testing helper functions...\n";
$helperFunctions = ['sendZForexMail', 'getMailSystem'];
$foundHelpers = 0;

foreach ($helperFunctions as $func) {
    if (function_exists($func)) {
        echo "   ✓ Function $func exists\n";
        $foundHelpers++;
    } else {
        echo "   ✗ Function $func not found\n";
    }
}

// Test 4: Test mail system instantiation
echo "\n4. Testing mail system instantiation...\n";
try {
    $mailSystem = getMailSystem();
    if ($mailSystem instanceof IntegratedMailSystem) {
        echo "   ✓ Mail system instantiated successfully\n";
        $instantiated = true;
    } else {
        echo "   ✗ Mail system instantiation failed\n";
        $instantiated = false;
    }
} catch (Exception $e) {
    echo "   ✗ Mail system instantiation error: " . $e->getMessage() . "\n";
    $instantiated = false;
}

// Test 5: Check mail system methods
echo "\n5. Testing mail system methods...\n";
$expectedMethods = [
    'sendRegisterVerification',
    'sendAfterVerifyCredentials', 
    'sendLoginAlert',
    'sendLiveAccountOpened',
    'sendKYCUpload',
    'sendKYCApproved',
    'sendDepositRequestSubmitted',
    'sendDepositApproved',
    'sendDepositFailed',
    'sendWithdrawalOTP',
    'sendWithdrawalRequestSubmitted',
    'sendWithdrawalApproved',
    'sendWithdrawalRejected',
    'sendIBCommissionWithdrawal',
    'sendReferralRegistration',
    'sendTradeOpened',
    'sendTradeClosed',
    'sendDailyTradeSummary',
    'sendPasswordReset',
    'sendPasswordChanged',
    'sendProfileUpdated',
    'sendMonthlyStatement'
];

$foundMethods = 0;
if ($instantiated) {
    foreach ($expectedMethods as $method) {
        if (method_exists($mailSystem, $method)) {
            echo "   ✓ Method $method exists\n";
            $foundMethods++;
        } else {
            echo "   ✗ Method $method not found\n";
        }
    }
} else {
    echo "   ✗ Cannot test methods - mail system not instantiated\n";
}

// Test 6: Test a sample email method (without actually sending)
echo "\n6. Testing sample email method...\n";
if ($instantiated && $foundMethods > 0) {
    try {
        // Test the sendRegisterVerification method structure
        $reflection = new ReflectionMethod($mailSystem, 'sendRegisterVerification');
        $parameters = $reflection->getParameters();
        echo "   ✓ sendRegisterVerification method signature:\n";
        foreach ($parameters as $param) {
            echo "     - " . $param->getName() . ($param->isOptional() ? ' (optional)' : ' (required)') . "\n";
        }
    } catch (Exception $e) {
        echo "   ✗ Error testing method: " . $e->getMessage() . "\n";
    }
}

// Test 7: Test configuration constants
echo "\n7. Testing mail configuration constants...\n";
$configConstants = [
    'EMAIL_AUTOMATION_ENABLED',
    'MAIL_HOST',
    'MAIL_USERNAME', 
    'MAIL_PASSWORD',
    'MAIL_PORT',
    'MAIL_ENCRYPTION',
    'MAIL_FROM_EMAIL',
    'MAIL_FROM_NAME'
];

$foundConstants = 0;
foreach ($configConstants as $constant) {
    if (defined($constant)) {
        echo "   ✓ Constant $constant is defined\n";
        $foundConstants++;
    } else {
        echo "   ✗ Constant $constant not defined\n";
    }
}

// Test 8: Test sendZForexMail function
echo "\n8. Testing sendZForexMail function...\n";
if (function_exists('sendZForexMail') && $instantiated) {
    try {
        $reflection = new ReflectionFunction('sendZForexMail');
        $parameters = $reflection->getParameters();
        echo "   ✓ sendZForexMail function signature:\n";
        foreach ($parameters as $param) {
            echo "     - " . $param->getName() . ($param->isOptional() ? ' (optional)' : ' (required)') . "\n";
        }
    } catch (Exception $e) {
        echo "   ✗ Error testing function: " . $e->getMessage() . "\n";
    }
}

// Summary
echo "\n=== TEST SUMMARY ===\n";
echo "Mail System Loading: " . ($mailSystemLoaded ? "✓ PASS" : "✗ FAIL") . "\n";
echo "Class Exists: " . ($classExists ? "✓ PASS" : "✗ FAIL") . "\n";
echo "Helper Functions: $foundHelpers/2 found\n";
echo "Instantiation: " . ($instantiated ? "✓ PASS" : "✗ FAIL") . "\n";
echo "Methods Found: $foundMethods/" . count($expectedMethods) . "\n";
echo "Config Constants: $foundConstants/" . count($configConstants) . "\n";

$totalTests = 4; // Main tests (loading, class, instantiation, basic functionality)
$passedTests = 0;

if ($mailSystemLoaded) $passedTests++;
if ($classExists) $passedTests++;
if ($foundHelpers >= 1) $passedTests++;
if ($instantiated) $passedTests++;

echo "\nOverall Result: $passedTests/$totalTests main tests passed\n";

if ($passedTests === $totalTests && $foundMethods >= 20) {
    echo "🎉 MAIL SYSTEM TEST: SUCCESS\n";
    echo "The mail system is properly configured and ready to use!\n";
    echo "\nNext Steps:\n";
    echo "1. Update unified_config.php with proper mail constants\n";
    echo "2. Test actual email sending with valid SMTP credentials\n";
    echo "3. Verify integration files are working correctly\n";
} else {
    echo "⚠️  MAIL SYSTEM TEST: ISSUES DETECTED\n";
    echo "Some components may need attention.\n";
}

echo "\n=== END TEST ===\n";
?>